import React from 'react';
import { RootInterpretation } from '../../services/api';
import { ELEMENT_COLORS } from '../../utils/constants';
import { Crown, Target, AlertTriangle, Heart, Briefcase, Sparkles } from 'lucide-react';

interface RootAnalysisProps {
  rootNumber: number;
  interpretation: RootInterpretation;
}

const RootAnalysis: React.FC<RootAnalysisProps> = ({ rootNumber, interpretation }) => {
  const getElementColor = (element: string) => {
    return ELEMENT_COLORS[element as keyof typeof ELEMENT_COLORS] || 'bg-gray-100 text-gray-800';
  };

  const analysisCards = [
    {
      icon: Crown,
      title: 'Core Traits',
      content: interpretation.traits,
      color: 'bg-purple-50 border-purple-200',
      iconColor: 'text-purple-600',
    },
    {
      icon: Target,
      title: 'Strengths',
      content: interpretation.strengths,
      color: 'bg-green-50 border-green-200',
      iconColor: 'text-green-600',
    },
    {
      icon: AlertTriangle,
      title: 'Challenges',
      content: interpretation.challenges,
      color: 'bg-orange-50 border-orange-200',
      iconColor: 'text-orange-600',
    },
    {
      icon: Heart,
      title: 'Desires',
      content: interpretation.desires,
      color: 'bg-pink-50 border-pink-200',
      iconColor: 'text-pink-600',
    },
    {
      icon: Briefcase,
      title: 'Career Guidance',
      content: interpretation.career,
      color: 'bg-blue-50 border-blue-200',
      iconColor: 'text-blue-600',
    },
  ];

  return (
    <div className="bg-white rounded-xl shadow-lg p-8">
      <div className="text-center mb-8">
        <h3 className="text-2xl font-bold text-gray-800 mb-4">Root Number Analysis</h3>
        
        {/* Root Number Display */}
        <div className="flex items-center justify-center space-x-6 mb-6">
          <div className="flex items-center space-x-3">
            <div className="w-16 h-16 bg-purple-600 text-white rounded-full flex items-center justify-center text-2xl font-bold shadow-lg">
              {rootNumber}
            </div>
            <div className="text-left">
              <div className="text-lg font-semibold text-gray-800">{interpretation.title}</div>
              <div className={`inline-block px-3 py-1 rounded-full text-sm font-medium ${getElementColor(interpretation.element)}`}>
                <Sparkles className="inline h-4 w-4 mr-1" />
                {interpretation.element}
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Analysis Cards Grid */}
      <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
        {analysisCards.map((card, index) => (
          <div
            key={index}
            className={`${card.color} border-2 rounded-lg p-6 transition-all duration-300 hover:shadow-md hover:scale-105`}
          >
            <div className="flex items-center space-x-3 mb-4">
              <div className={`p-2 rounded-lg bg-white ${card.iconColor}`}>
                <card.icon className="h-5 w-5" />
              </div>
              <h4 className="font-semibold text-gray-800">{card.title}</h4>
            </div>
            <p className="text-gray-700 text-sm leading-relaxed">
              {card.content}
            </p>
          </div>
        ))}
      </div>

      {/* Element Description */}
      <div className="mt-8 p-6 bg-gradient-to-r from-gray-50 to-gray-100 rounded-lg">
        <div className="flex items-center space-x-2 mb-3">
          <Sparkles className="h-5 w-5 text-purple-600" />
          <h4 className="text-lg font-semibold text-gray-800">Your Element: {interpretation.element}</h4>
        </div>
        <p className="text-gray-700 text-sm">
          Your root number {rootNumber} is associated with the {interpretation.element} element, which influences your 
          natural tendencies, decision-making style, and optimal business environment. This element guides your 
          approach to relationships, career choices, and personal growth opportunities.
        </p>
      </div>

      {/* Quick Summary */}
      <div className="mt-6 p-4 bg-purple-600 text-white rounded-lg">
        <h4 className="font-semibold mb-2">Quick Summary</h4>
        <p className="text-sm opacity-90">
          As a Root Number {rootNumber} ({interpretation.element}), you excel in {interpretation.strengths.toLowerCase()}. 
          Focus on {interpretation.career.toLowerCase()} to maximize your potential.
        </p>
      </div>
    </div>
  );
};

export default RootAnalysis;